UNIT GLScene;

{ TGLScene    - An encapsulation of the OpenGL API
  Version     - 0.3.2
  Last Change - 26 June 1997
  for more information see help file
}

INTERFACE

USES Classes, ComCtrls, CSG, GL, GLU, Graphics, Messages, GLTexture, Windows;

TYPE TObjectHandle      = GLUInt; // aquivalent to a display list ID or
                                  // GL_LIGHT0 (...) constants

     // used to describe what kind of winding has a front face
     TFaceWinding       = (fwClockWise,fwCounterClockWise);

     // used to reflect all relevant (binary) states of OpenGL subsystem
     TGLState           = (stContextValid,stContextActive,stAlphaTest,stAutoNormal,
                           stBlend,stColorMaterial,stCullFace,stDepthTest,stDither,
                           stFog,stLighting,stLineSmooth,stLineStipple,
                           stLogicOp,stNormalize,stPointSmooth,stPolygonSmooth,
                           stPolygonStipple,stScissorTest,stStencilTest);
     TGLStates          = SET OF TGLState;

     {used to decribe only the changes in an object, which have to be
      reflected in the scene}
     TObjectChange      = (ocPosition,ocRotation,ocScaling,
                           ocShininess, ocSpot,ocAttenuation);
     TObjectChanges     = SET OF TObjectChange;

     {used to decribe only the global changes in the scene, which have to be
      reflected in the scene}
     TSceneChange       = (scBackground);
     TSceneChanges      = SET OF TSceneChange;

     // flags for design notification
     TSceneOperation    = (soAdd,soRemove,soMove,soRename);

     // flags for allocated buffers
     TBuffer            = (buColor,buDepth,buStencil,buAccum,buAux);
     TBuffers           = SET OF TBuffer;

     TVertex            = TAffineFloatVector;
     PVertex            = ^TVertex;
     TNormalVector      = TAffineFloatVector;
     PNormalVector      = ^TNormalVector;
     TColorVector       = THomogenFloatVector;

     TGLCoordinates     = RECORD
                            CASE Boolean OF
                              True  : (Coordinates: THomogenFloatVector);
                              False : (X, Y, Z, W: GLFloat);
                          END;
     TRectangle         = RECORD
                            Left, Top, Width, Height: Integer;
                          END;

     TBoundingBox       = RECORD
                            LeftLowerFront,
                            RightUpperBack  : TAffineFloatVector;
                          END;

     TSceneObject = CLASS;
     TComposite = CLASS;
     TGLScene   = CLASS;

     TGLColor = CLASS(TPersistent)
     PRIVATE
       FParent  : TSceneObject;
       FChanged : Boolean;
       FColor   : TColorVector;
       PROCEDURE SetAlpha(AValue: GLFloat);
       PROCEDURE SetBlue(AValue: GLFloat);
       PROCEDURE SetColor(AColor: TColorVector);
       PROCEDURE SetGreen(AValue: GLFloat);
       PROCEDURE SetRed(AValue: GLFloat);
     PUBLIC
       CONSTRUCTOR Create(AParent: TSceneObject);
       PROPERTY Color: TColorVector READ FColor WRITE SetColor;
     PUBLISHED
       PROPERTY Red:   GLFloat READ FColor[0] WRITE SetRed STORED True;
       PROPERTY Green: GLFloat READ FColor[1] WRITE SetGreen STORED True;
       PROPERTY Blue:  GLFloat READ FColor[2] WRITE SetBlue STORED True;
       PROPERTY Alpha: GLFloat READ FColor[3] WRITE SetAlpha STORED True;
     END;

     TSceneObject = CLASS(TComponent)
     PRIVATE
       FHandle      : TObjectHandle;
       FPosition    : TGLCoordinates;
       FRotation,
       FScaling     : TAffineFloatVector;
       FChanged     : Boolean;
       FChanges     : TObjectChanges;
       FAmbient,
       FDiffuse,
       FSpecular,
       FEmission    : TGLColor;
       FShininess   : GLFloat;
       FParent      : TComposite;
       FGLScene     : TGLScene;
       FBoundingBox : TBoundingBox;
       FCSGGroup    : TCSGGroup;
       FTexture     : TTexture;
       FUNCTION  GetHandle: TObjectHandle;
       FUNCTION  GetIndex: Integer;
       PROCEDURE SetAmbient(AValue: TGLColor);
       PROCEDURE SetChanged(AValue: Boolean);
       PROCEDURE SetCSGGroup(AGroup: TCSGGroup);
       PROCEDURE SetDiffuse(AValue: TGLColor);
       PROCEDURE SetIndex(AValue: Integer);
       PROCEDURE SetShininess(AValue: GLFloat);
       PROCEDURE SetSpecular(AValue: TGLColor);
       PROCEDURE SetPosition(APosition: TGLCoordinates);
       PROCEDURE SetPositionW(AValue: GLFloat);
       PROCEDURE SetPositionX(AValue: GLFloat);
       PROCEDURE SetPositionY(AValue: GLFloat);
       PROCEDURE SetPositionZ(AValue: GLFloat);
       PROCEDURE SetRotationX(AValue: GLFloat);
       PROCEDURE SetRotationY(AValue: GLFloat);
       PROCEDURE SetRotationZ(AValue: GLFloat);
     PROTECTED
       FUNCTION GetParentComponent: TComponent; OVERRIDE;
       FUNCTION HasParent: Boolean; OVERRIDE;
       PROCEDURE SetName(CONST NewName: TComponentName); OVERRIDE;
       PROCEDURE SetParentComponent(Value: TComponent); OVERRIDE;
     PUBLIC
       CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
       DESTRUCTOR Destroy; OVERRIDE;

       PROCEDURE BuildList; VIRTUAL; ABSTRACT;
       PROCEDURE DestroyList; VIRTUAL;
       PROCEDURE FinishList; VIRTUAL;
       PROCEDURE MoveTo(NewParent: TComposite); VIRTUAL;
       PROCEDURE PrepareList; VIRTUAL;
       PROCEDURE Rotate(Rx,Ry,Rz : GLFloat); VIRTUAL;
       PROCEDURE Scale(Sx,Sy,Sz : GLFloat); VIRTUAL;
       PROCEDURE Translate(Tx,Ty,Tz : GLFloat); VIRTUAL;

       PROPERTY Ambient: TGLColor READ FAmbient WRITE SetAmbient;
       PROPERTY BoundingBox: TBoundingBox READ FBoundingBox;
       PROPERTY Changed: Boolean READ FChanged WRITE SetChanged DEFAULT False;
       PROPERTY CSGGroup: TCSGGroup READ FCSGGroup WRITE SetCSGGroup;
       PROPERTY Diffuse: TGLColor READ FDiffuse WRITE SetDiffuse;
       PROPERTY RotationX: GLFloat READ FRotation[0] WRITE SetRotationX;
       PROPERTY RotationY: GLFloat READ FRotation[1] WRITE SetRotationY;
       PROPERTY RotationZ: GLFloat READ FRotation[2] WRITE SetRotationZ;
       PROPERTY Handle: TObjectHandle READ GetHandle;
       PROPERTY Index: Integer READ GetIndex WRITE SetIndex;
       PROPERTY Parent: TComposite READ FParent;
       PROPERTY Position: TGLCoordinates READ FPosition WRITE SetPosition;
       PROPERTY Scene: TGLScene READ FGLScene;
       PROPERTY Shininess: GLFloat READ FShininess WRITE SetShininess;
       PROPERTY Specular: TGLColor READ FSpecular WRITE SetSpecular;
       PROPERTY Texture: TTexture READ FTexture WRITE FTexture;
       PROPERTY W: GLFloat READ FPosition.W WRITE SetPositionW;
       PROPERTY X: GLFloat READ FPosition.X WRITE SetPositionX;
       PROPERTY Y: GLFloat READ FPosition.Y WRITE SetPositionY;
       PROPERTY Z: GLFloat READ FPosition.Z WRITE SetPositionZ;
     END;

     TProxyObject = CLASS(TSceneObject)
     PRIVATE
       FObjectLink : TSceneObject;
     PUBLIC
       PROPERTY ObjectLink: TSceneObject READ FObjectLink WRITE FObjectLink;
     END;

     TExternalObject = CLASS(TSceneObject)
     PRIVATE
       FTypeName : STRING;
     PUBLIC
       PROPERTY TypeName: STRING READ FTypeName WRITE FTypeName;
     END;

     TLightSource = CLASS(TSceneObject)
     PRIVATE
       FSpotDirection        : TAffineFloatVector;
       FSpotExponent,
       FSpotCutOff,
       FConstAttenuation,
       FLinearAttenuation,
       FQuadraticAttenuation : GLFloat;
       FShining              : Boolean;
       FUNCTION  GetLightID: TObjectHandle;
       PROCEDURE SetConstAttenuation(AValue: GLFloat);
       PROCEDURE SetLinearAttenuation(AValue: GLFloat);
       PROCEDURE SetQuadraticAttenuation(AValue: GLFloat);
       PROCEDURE SetShining(AValue: Boolean);
       PROCEDURE SetSpotDirection(AVector: TAffineFloatVector);
       PROCEDURE SetSpotExponent(AValue: GLFloat);
       PROCEDURE SetSpotCutOff(AValue: GLFloat);
     PROTECTED
     PUBLIC
       CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
       PROCEDURE DestroyList; OVERRIDE;
       PROCEDURE Release;
       PROCEDURE SwitchOn;
       PROCEDURE SwitchOff;

       PROPERTY LightID: TObjectHandle READ GetLightID;
       PROPERTY SpotDirection: TAffineFloatVector READ FSpotDirection WRITE SetSpotDirection;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY ConstAttenuation: GLFloat READ FConstAttenuation WRITE SetConstAttenuation;
       PROPERTY Diffuse;
       PROPERTY LinearAttenuation: GLFloat READ FLinearAttenuation WRITE SetLinearAttenuation;
       PROPERTY QuadraticAttenuation: GLFloat READ FQuadraticAttenuation WRITE SetQuadraticAttenuation;
       PROPERTY Shining: Boolean READ FShining WRITE SetShining;
       PROPERTY Specular;
       PROPERTY SpotCutOff: GLFloat READ FSpotCutOff WRITE SetSpotCutOff;
       PROPERTY SpotExponent: GLFloat READ FSpotExponent WRITE SetSpotExponent;
       PROPERTY W;
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TCamera = CLASS(TSceneObject)
     PRIVATE
       FActive : Boolean;
       PROCEDURE SetActive(AValue: Boolean);
     PUBLISHED
       PROPERTY Active: Boolean READ FActive WRITE SetActive;
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TSceneObjectClass = CLASS OF TSceneObject;

     TComposite = CLASS(TSceneObject)
     PRIVATE
       FChildren: TList;
       FCSGOperation : TCSGOperation;
       FUNCTION Get(Index: Integer): TSceneObject;
       FUNCTION GetCount: Integer;
       PROCEDURE SetCSGOperation(AValue: TCSGOperation);
     PROTECTED
       PROCEDURE GetChildren(AProc: TGetChildProc); OVERRIDE;
       PROCEDURE SetChildOrder(AChild: TComponent; Order: Integer); OVERRIDE;
     PUBLIC
       CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
       DESTRUCTOR Destroy; OVERRIDE;

       FUNCTION AddNewChild(AChild: TSceneObjectClass): TSceneObject;

       PROCEDURE AddChild(AChild: TSceneObject);
       PROCEDURE BuildList; OVERRIDE;
       PROCEDURE DeleteChildren;
       PROCEDURE Insert(AIndex: Integer; AChild: TSceneObject);
       PROCEDURE PrepareList; OVERRIDE;
       PROCEDURE Remove(AChild: TSceneObject; KeepChildren: Boolean);

       PROPERTY Count: Integer read GetCount;
       PROPERTY Children[Index: Integer]: TSceneObject READ Get; DEFAULT;
     PUBLISHED
       PROPERTY Ambient;
       PROPERTY CSGOperation: TCSGOperation READ FCSGOperation WRITE SetCSGOperation;
       PROPERTY Diffuse;
       PROPERTY RotationX;
       PROPERTY RotationY;
       PROPERTY RotationZ;
       PROPERTY Shininess;
       PROPERTY Specular;
       PROPERTY X;
       PROPERTY Y;
       PROPERTY Z;
     END;

     TAbstractSceneTree = CLASS(TCustomTreeView)
     PROTECTED
       PROCEDURE Notify(AObject: TSceneObject; Operation: TSceneOperation); VIRTUAL; ABSTRACT;
     END;

     TGLScene = CLASS(TCustomControl)
     PRIVATE
       // handles
       FRenderingContext : HGLRC;

       // OpenGL properties
       FMaxLightSources  : GLInt;
       FDoubleBuffered,
       FAccelerated,
       FDepthTest,
       FFaceCulling,
       FLighting         : Boolean;
       FCurrentStates    : TGLStates;
       FFrontFaceWinding : TFaceWinding;
       FBackground       : TColor;

       // private variables
       FFrames           : Longint;              // used to perform monitoring
       FTicks            : Longint;              // used to perform monitoring
       FSceneChanges     : TSceneChanges;
       FMonitoring,
       FLightChanged,
       FObjectChanged    : Boolean;
       FUpdateCount      : Integer;
       FFramesPerSecond  : GLFloat;
       FViewPort         : TRectangle;
       FFOVAngle         : GLFLoat;
       FNearClipPlane    : GLFLoat;
       FFarClipPlane     : GLFLoat;
       FObjects,
       FLightSources,
       FCameras          : TComposite;
       FNotifiers        : TList;
       FBuffers          : TBuffers;
       PROCEDURE RegisterExtensions(Extensions: STRING);
       PROCEDURE SetBackground(AColor: TColor);
       PROCEDURE CreateParams(VAR Params: TCreateParams); OVERRIDE;
       FUNCTION  GetRenderingContext: HGLRC;
       PROCEDURE SetDCPixelFormat(Handle: HDC);
       PROCEDURE SetDepthTest(AValue: Boolean);
       PROCEDURE SetFaceCulling(AValue: Boolean);
       PROCEDURE SetFrontFaceWinding(AValue: TFaceWinding);
       PROCEDURE SetLighting(AValue: Boolean);
       PROCEDURE WMEraseBkgnd(VAR Message: TWMEraseBkgnd); MESSAGE WM_ERASEBKGND;
       PROCEDURE WMSize(VAR Message: TWMSize); MESSAGE WM_SIZE;
     PROTECTED
       PROCEDURE AdjustLightSources;
       PROCEDURE ApplyCamera;
       PROCEDURE GetChildren(AProc: TGetChildProc); OVERRIDE;
       FUNCTION  GetDeviceContext(VAR WindowHandle: HWnd): HDC; OVERRIDE;
       PROCEDURE Loaded; OVERRIDE;
       PROCEDURE Note(AObject: TSceneObject; Operation: TSceneOperation);
       PROCEDURE RequestedState(States: TGLStates);
       PROCEDURE ReadContextProperties;
       PROCEDURE Paint; OVERRIDE;
       PROCEDURE SetChildOrder(AChild: TComponent; Order: Integer); OVERRIDE;
       PROCEDURE UnnecessaryState(States: TGLStates);
     PUBLIC
       CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
       DESTRUCTOR  Destroy; OVERRIDE;
       PROCEDURE AddNotifier(ASceneTree: TAbstractSceneTree);
       PROCEDURE BeginUpdate;
       PROCEDURE DrawScene;
       PROCEDURE EndUpdate;
       FUNCTION  IsUpdating: Boolean;
       PROCEDURE NotifyChange;
       PROCEDURE RemoveNotifier(ASceneTree: TAbstractSceneTree);
       PROCEDURE SetPerspective(AFOVAngle,ANearPlane,AFarPlane: GLFloat);
       PROCEDURE SetViewPort(X,Y,W,H: Integer);

       PROPERTY Buffers: TBuffers READ FBuffers;
       PROPERTY Cameras: TComposite READ FCameras;
       PROPERTY FramesPerSecond: GLFloat READ FFramesPerSecond;
       PROPERTY LightSources: TComposite READ FLightSources;
       PROPERTY MaxLightSources: Integer READ FMaxLightSources;
       PROPERTY Objects: TComposite READ FObjects;
       PROPERTY RenderingContext: HGLRC READ GetRenderingContext;
     PUBLISHED
       PROPERTY Align;
       PROPERTY Background: TColor READ FBackground WRITE SetBackground;
       PROPERTY DepthTest: Boolean READ FDepthTest WRITE SetDepthTest;
       PROPERTY EnableMonitoring: Boolean READ FMonitoring WRITE FMonitoring;
       PROPERTY FaceCulling: Boolean READ FFaceCulling WRITE SetFaceCulling;
       PROPERTY FarClipPlane: GLFloat READ FFarClipPlane WRITE FFarClipPlane;
       PROPERTY FrontFaceWinding: TFaceWinding READ FFrontFaceWinding WRITE SetFrontFaceWinding;
       PROPERTY Lighting: Boolean READ FLighting WRITE SetLighting;
       PROPERTY NearClipPlane: GLFloat READ FNearClipPlane WRITE FNearClipPlane;
     END;

     PROCEDURE CheckOpenGLError;
     FUNCTION  MakeAffineVector(X,Y,Z: GLFloat) : TAffineFloatVector;
     FUNCTION  MakeHomogenVector(X,Y,Z,W: GLFloat) : THomogenFloatVector;

CONST
//  color definitions
  // sort of grays
  ColorGray05            : TColorVector = (0.05, 0.05, 0.05, 1);
  ColorGray10            : TColorVector = (0.10, 0.10, 0.10, 1);
  ColorGray15            : TColorVector = (0.15, 0.15, 0.15, 1);
  ColorGray20            : TColorVector = (0.20, 0.20, 0.20, 1);
  ColorGray25            : TColorVector = (0.25, 0.25, 0.25, 1);
  ColorGray30            : TColorVector = (0.30, 0.30, 0.30, 1);
  ColorGray35            : TColorVector = (0.35, 0.35, 0.35, 1);
  ColorGray40            : TColorVector = (0.40, 0.40, 0.40, 1);
  ColorGray45            : TColorVector = (0.45, 0.45, 0.45, 1);
  ColorGray50            : TColorVector = (0.50, 0.50, 0.50, 1);
  ColorGray55            : TColorVector = (0.55, 0.55, 0.55, 1);
  ColorGray60            : TColorVector = (0.60, 0.60, 0.60, 1);
  ColorGray65            : TColorVector = (0.65, 0.65, 0.65, 1);
  ColorGray70            : TColorVector = (0.70, 0.70, 0.70, 1);
  ColorGray75            : TColorVector = (0.75, 0.75, 0.75, 1);
  ColorGray80            : TColorVector = (0.80, 0.80, 0.80, 1);
  ColorGray85            : TColorVector = (0.85, 0.85, 0.85, 1);
  ColorGray90            : TColorVector = (0.90, 0.90, 0.90, 1);
  ColorGray95            : TColorVector = (0.95, 0.95, 0.95, 1);
  ColorWhite             : TColorVector = (1,    1,    1,    1);

  // other grays
  ColorDimGray           : TColorVector = (0.329412, 0.329412, 0.329412, 1);
  ColorDimGrey           : TColorVector = (0.329412, 0.329412, 0.329412, 1);
  ColorGray              : TColorVector = (0.752941, 0.752941, 0.752941, 1);
  ColorGrey              : TColorVector = (0.752941, 0.752941, 0.752941, 1);
  ColorLightGray         : TColorVector = (0.658824, 0.658824, 0.658824, 1);
  ColorLightGrey         : TColorVector = (0.658824, 0.658824, 0.658824, 1);
  ColorVLightGray        : TColorVector = (0.80,     0.80,     0.80,     1);
  ColorVLightGrey        : TColorVector = (0.80,     0.80,     0.80,     1);

  // colors en masse
  ColorAquamarine        : TColorVector = (0.439216, 0.858824, 0.576471, 1);
  ColorBlueViolet        : TColorVector = (0.62352,  0.372549, 0.623529, 1);
  ColorBrown             : TColorVector = (0.647059, 0.164706, 0.164706, 1);
  ColorCadetBlue         : TColorVector = (0.372549, 0.623529, 0.623529, 1);
  ColorCoral             : TColorVector = (1,        0.498039, 0.0,      1);
  ColorCornflowerBlue    : TColorVector = (0.258824, 0.258824, 0.435294, 1);
  ColorDarkGreen         : TColorVector = (0.184314, 0.309804, 0.184314, 1);
  ColorDarkOliveGreen    : TColorVector = (0.309804, 0.309804, 0.184314, 1);
  ColorDarkOrchid        : TColorVector = (0.6,      0.196078, 0.8,      1);
  ColorDarkSlateBlue     : TColorVector = (0.419608, 0.137255, 0.556863, 1);
  ColorDarkSlateGray     : TColorVector = (0.184314, 0.309804, 0.309804, 1);
  ColorDarkSlateGrey     : TColorVector = (0.184314, 0.309804, 0.309804, 1);
  ColorDarkTurquoise     : TColorVector = (0.439216, 0.576471, 0.858824, 1);
  ColorFirebrick         : TColorVector = (0.556863, 0.137255, 0.137255, 1);
  ColorForestGreen       : TColorVector = (0.137255, 0.556863, 0.137255, 1);
  ColorGold              : TColorVector = (0.8,      0.498039, 0.196078, 1);
  ColorGoldenrod         : TColorVector = (0.858824, 0.858824, 0.439216, 1);
  ColorGreenYellow       : TColorVector = (0.576471, 0.858824, 0.439216, 1);
  ColorIndian            : TColorVector = (0.309804, 0.184314, 0.184314, 1);
  ColorKhaki             : TColorVector = (0.623529, 0.623529, 0.372549, 1);
  ColorLightBlue         : TColorVector = (0.74902,  0.847059, 0.847059, 1);
  ColorLightSteelBlue    : TColorVector = (0.560784, 0.560784, 0.737255, 1);
  ColorLimeGreen         : TColorVector = (0.196078, 0.8,      0.196078, 1);
  ColorMaroon            : TColorVector = (0.556863, 0.137255, 0.419608, 1);
  ColorMediumAquamarine  : TColorVector = (0.196078, 0.8,      0.6,      1);
  ColorMediumBlue        : TColorVector = (0.196078, 0.196078, 0.8,      1);
  ColorMediumForestGreen : TColorVector = (0.419608, 0.556863, 0.137255, 1);
  ColorMediumGoldenrod   : TColorVector = (0.917647, 0.917647, 0.678431, 1);
  ColorMediumOrchid      : TColorVector = (0.576471, 0.439216, 0.858824, 1);
  ColorMediumSeaGreen    : TColorVector = (0.258824, 0.435294, 0.258824, 1);
  ColorMediumSlateBlue   : TColorVector = (0.498039, 0,        1,        1);
  ColorMediumSpringGreen : TColorVector = (0.498039, 1,        0,        1);
  ColorMediumTurquoise   : TColorVector = (0.439216, 0.858824, 0.858824, 1);
  ColorMediumViolet      : TColorVector = (0.858824, 0.439216, 0.576471, 1);
  ColorMidnightBlue      : TColorVector = (0.184314, 0.184314, 0.309804, 1);
  ColorNavy              : TColorVector = (0.137255, 0.137255, 0.556863, 1);
  ColorNavyBlue          : TColorVector = (0.137255, 0.137255, 0.556863, 1);
  ColorOrange            : TColorVector = (1,        0.5,      0.0,      1);
  ColorOrangeRed         : TColorVector = (1,        0.25,     0,        1);
  ColorOrchid            : TColorVector = (0.858824, 0.439216, 0.858824, 1);
  ColorPaleGreen         : TColorVector = (0.560784, 0.737255, 0.560784, 1);
  ColorPink              : TColorVector = (0.737255, 0.560784, 0.560784, 1);
  ColorPlum              : TColorVector = (0.917647, 0.678431, 0.917647, 1);
  ColorSalmon            : TColorVector = (0.435294, 0.258824, 0.258824, 1);
  ColorSeaGreen          : TColorVector = (0.137255, 0.556863, 0.419608, 1);
  ColorSienna            : TColorVector = (0.556863, 0.419608, 0.137255, 1);
  ColorSkyBlue           : TColorVector = (0.196078, 0.6,      0.8,      1);
  ColorSlateBlue         : TColorVector = (0,        0.498039, 1,        1);
  ColorSpringGreen       : TColorVector = (0,        1,        0.498039, 1);
  ColorSteelBlue         : TColorVector = (0.137255, 0.419608, 0.556863, 1);
  ColorTan               : TColorVector = (0.858824, 0.576471, 0.439216, 1);
  ColorThistle           : TColorVector = (0.847059, 0.74902,  0.847059, 1);
  ColorTurquoise         : TColorVector = (0.678431, 0.917647, 0.917647, 1);
  ColorViolet            : TColorVector = (0.309804, 0.184314, 0.309804, 1);
  ColorVioletRed         : TColorVector = (0.8,      0.196078, 0.6,      1);
  ColorWheat             : TColorVector = (0.847059, 0.847059, 0.74902,  1);
  ColorYellowGreen       : TColorVector = (0.6,      0.8,      0.196078, 1);
  ColorSummerSky         : TColorVector = (0.22,     0.69,     0.87,     1);
  ColorRichBlue          : TColorVector = (0.35,     0.35,     0.67,     1);
  ColorBrass             : TColorVector = (0.71,     0.65,     0.26,     1);
  ColorCopper            : TColorVector = (0.72,     0.45,     0.20,     1);
  ColorBronze            : TColorVector = (0.55,     0.47,     0.14,     1);
  ColorBronze2           : TColorVector = (0.65,     0.49,     0.24,     1);
  ColorSilver            : TColorVector = (0.90,     0.91,     0.98,     1);
  ColorBrightGold        : TColorVector = (0.85,     0.85,     0.10,     1);
  ColorOldGold           : TColorVector = (0.81,     0.71,     0.23,     1);
  ColorFeldspar          : TColorVector = (0.82,     0.57,     0.46,     1);
  ColorQuartz            : TColorVector = (0.85,     0.85,     0.95,     1);
  ColorNeonPink          : TColorVector = (1.00,     0.43,     0.78,     1);
  ColorDarkPurple        : TColorVector = (0.53,     0.12,     0.47,     1);
  ColorNeonBlue          : TColorVector = (0.30,     0.30,     1.00,     1);
  ColorCoolCopper        : TColorVector = (0.85,     0.53,     0.10,     1);
  ColorMandarinOrange    : TColorVector = (0.89,     0.47,     0.20,     1);
  ColorLightWood         : TColorVector = (0.91,     0.76,     0.65,     1);
  ColorMediumWood        : TColorVector = (0.65,     0.50,     0.39,     1);
  ColorDarkWood          : TColorVector = (0.52,     0.37,     0.26,     1);
  ColorSpicyPink         : TColorVector = (1.00,     0.11,     0.68,     1);
  ColorSemiSweetChoc     : TColorVector = (0.42,     0.26,     0.15,     1);
  ColorBakersChoc        : TColorVector = (0.36,     0.20,     0.09,     1);
  ColorFlesh             : TColorVector = (0.96,     0.80,     0.69,     1);
  ColorNewTan            : TColorVector = (0.92,     0.78,     0.62,     1);
  ColorNewMidnightBlue   : TColorVector = (0.00,     0.00,     0.61,     1);
  ColorVeryDarkBrown     : TColorVector = (0.35,     0.16,     0.14,     1);
  ColorDarkBrown         : TColorVector = (0.36,     0.25,     0.20,     1);
  ColorDarkTan           : TColorVector = (0.59,     0.41,     0.31,     1);
  ColorGreenCopper       : TColorVector = (0.32,     0.49,     0.46,     1);
  ColorDkGreenCopper     : TColorVector = (0.29,     0.46,     0.43,     1);
  ColorDustyRose         : TColorVector = (0.52,     0.39,     0.39,     1);
  ColorHuntersGreen      : TColorVector = (0.13,     0.37,     0.31,     1);
  ColorScarlet           : TColorVector = (0.55,     0.09,     0.09,     1);
  ColorMediumPurple      : TColorVector = (0.73,     0.16,     0.96,     1);
  ColorLightPurple       : TColorVector = (0.87,     0.58,     0.98,     1);
  ColorVeryLightPurple   : TColorVector = (0.94,     0.81,     0.99,     1);

//------------------------------------------------------------------------------

IMPLEMENTATION

USES Consts, Dialogs, DsgnIntf, Forms, GLObjects, GLTree, SysUtils;

{$R GLScene.RES}

CONST FaceWindingToNative : ARRAY[fwClockWise..fwCounterClockWise] OF GLEnum = (GL_CW,GL_CCW);
      NativeToFaceWinding : ARRAY[GL_CW..GL_CCW] OF TFaceWinding = (fwClockWise,fwCounterClockWise);

      GLAllStates = [stContextValid..stStencilTest];

      // Resource ID's:
      ErrorNoContext      = 104;

TYPE OpenGLError = CLASS(Exception);

VAR LangOffset : Word = 0;

//------------------ external global routines ----------------------------------

PROCEDURE CheckOpenGLError;

{Gets the oldest error from OpenGL engine and tries to clear the error queue.
 Because under some circumstances reading the error code creates a new error
 and thus hanging up the thread, we limit the loop to 6 reads.}

VAR GLError : GLEnum;
    Count   : Word;

BEGIN
  GLError:=glGetError;
  IF GLError <> GL_NO_ERROR THEN
  BEGIN
    Count:=0;
    WHILE (glGetError <> GL_NO_ERROR) AND (Count < 6) DO Inc(Count);
    RAISE OpenGLError.Create(StrPas(PChar(gluErrorString(GLError))));
  END;
END;

//------------------------------------------------------------------------------

FUNCTION MakeAffineVector(X,Y,Z: GLFloat) : TAffineFloatVector;

BEGIN
  Result[0]:=X; Result[1]:=Y; Result[2]:=Z;
END;

//------------------------------------------------------------------------------

FUNCTION MakeHomogenVector(X,Y,Z,W: GLFloat) : THomogenFloatVector;

BEGIN
  Result[0]:=X; Result[1]:=Y; Result[2]:=Z; Result[3]:=W;
END;

//------------------ internal global routines ----------------------------------

PROCEDURE ShowError(Msg: Word);

BEGIN
  MessageBeep(MB_ICONHAND);
  RAISE OpenGLError.CreateRes(Msg+LangOffset);
END;

//------------------------------------------------------------------------------

PROCEDURE ShowErrorFormatted(Msg: Word; CONST Args: ARRAY OF CONST);

BEGIN
  MessageBeep(MB_ICONHAND);
  RAISE OpenGLError.CreateResFmt(Msg+LangOffset,Args);
END;

//------------------------------------------------------------------------------

PROCEDURE ConvertWinColor(AColor: TColor; VAR red, green, blue: GLFloat);

// converts a delphi color into its rgb fragments and correct range

VAR WinColor : Longint;

BEGIN
  // delphi color to Windows color
  WinColor:=ColorToRGB(AColor);
  // convert 0..255 range into 0..1 range
  red:=(WinColor AND $FF)/256;
  green:=((WinColor SHR 8) AND $FF)/256;
  blue:=((WinColor SHR 16) AND $FF)/256;
END;

//------------------------------------------------------------------------------

FUNCTION TestName(AOwner: TComponent; CONST AName: STRING): Boolean;

BEGIN
  Result:=(AOwner = NIL) OR (AOwner.FindComponent(AName) = NIL);
END;

//------------------------------------------------------------------------------

FUNCTION CreateSceneObject(AScene: TGLScene; AObject: TSceneObjectClass): TSceneObject;

{Creates a new component with unique name. This function is internally used by functions
 which need to create new scene objects (AddNewChild...).}

VAR AOwner : TForm;
    I      : Integer;
    AName  : STRING;

BEGIN
  // get the owner of the parent
  AOwner:=GetParentForm(AScene);
  {Create the object with the resulting form as owner (at design time, this object
   will then appear in the source code.}
  Result:=AObject.Create(AOwner);
  // create an unique name
  AName:=Copy(AObject.ClassName,2,255);
  FOR I:=1 TO High(Integer) DO
    IF TestName(AOwner,AName+IntToStr(I)) THEN
    BEGIN
      Result.Name:=AName+IntToStr(I);
      Exit;
    END;
  Result.Free;  
  RAISE Exception.Create('Cannot create unique name for '+AObject.ClassName);
END;

//---------------------- TGLColor ----------------------------------------------

CONSTRUCTOR TGLColor.Create(AParent: TSceneObject);

BEGIN
  INHERITED Create;
  FParent:=AParent;
  FChanged:=False;
  FColor[0]:=0; FColor[1]:=0; FColor[2]:=0; FColor[3]:=1;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLColor.SetAlpha(AValue: GLFloat);

BEGIN
  IF FColor[3] <> AValue THEN
  BEGIN
    FColor[3]:=AValue;
    FChanged:=True;
    FParent.Changed:=True;
    FParent.FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLColor.SetBlue(AValue: GLFloat);

BEGIN
  IF FColor[2] <> AValue THEN
  BEGIN
    FColor[2]:=AValue;
    FChanged:=True;
    FParent.Changed:=True;
    FParent.FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLColor.SetColor(AColor: TColorVector);

BEGIN
  FColor:=AColor;
  FChanged:=True;
  FParent.Changed:=True;
  FParent.FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLColor.SetGreen(AValue: GLFloat);

BEGIN
  IF FColor[1] <> AValue THEN
  BEGIN
    FColor[1]:=AValue;
    FChanged:=True;
    FParent.Changed:=True;
    FParent.FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLColor.SetRed(AValue: GLFloat);

BEGIN
  IF FColor[0] <> AValue THEN
  BEGIN
    FColor[0]:=AValue;
    FChanged:=True;
    FParent.Changed:=True;
    FParent.FGLScene.NotifyChange;
  END;
END;


//------------------ TSceneObject ----------------------------------------------

CONSTRUCTOR TSceneObject.Create(AOwner: TComponent);

BEGIN
  INHERITED Create(AOwner);
  FHandle:=0;
  FChanged:=False;
  FChanges:=[];
  FGLScene:=NIL;
  FParent:=NIL;
  FPosition.X:=0; FPosition.Y:=0; FPosition.Z:=0;
  FRotation[0]:=0; FRotation[1]:=0; FRotation[2]:=0;
  FScaling[0]:=1; FScaling[1]:=1; FScaling[2]:=1;
  FAmbient:=TGLColor.Create(Self);
  FDiffuse:=TGLColor.Create(Self);
  FSpecular:=TGLColor.Create(Self);
  FTexture:=TTexture.Create(Self);
END;

//------------------------------------------------------------------------------

DESTRUCTOR TSceneObject.Destroy;

BEGIN
  FTexture.Free;
  FAmbient.Free;
  FDiffuse.Free;
  FSpecular.Free;
  DestroyList;
  IF assigned(FParent) THEN FParent.Remove(Self,False);
  IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  INHERITED Destroy;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.DestroyList;

BEGIN
  IF FHandle > 0 THEN
  BEGIN
    glDeleteLists(FHandle,1);
    FHandle:=0;
  END;
END;

//------------------------------------------------------------------------------

FUNCTION TSceneObject.GetHandle: TObjectHandle;

BEGIN
  IF (FHandle = 0) OR FChanged THEN
  BEGIN
    IF FHandle = 0 THEN FHandle:=glGenLists(1);
    PrepareList;
    BuildList;
    FinishList;
    FChanged:=False;
  END;
  Result:=FHandle;
END;

//------------------------------------------------------------------------------

FUNCTION TSceneObject.GetIndex: Integer;

BEGIN
  Result:=-1;
  IF assigned(FParent) THEN Result:=FParent.FChildren.IndexOf(Self);
END;

//------------------------------------------------------------------------------

FUNCTION TSceneObject.GetParentComponent: TComponent;

BEGIN
  Result:=FParent;
END;

//------------------------------------------------------------------------------

FUNCTION TSceneObject.HasParent: Boolean;

BEGIN
  Result:=True;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetName(CONST NewName: TComponentName);

BEGIN
  IF Name <> NewName THEN
  BEGIN
    INHERITED SetName(NewName);
    IF assigned(FGLScene) THEN FGLScene.Note(Self,soRename);
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetIndex(AValue: Integer);

VAR Count   : Integer;
    AParent : TComposite;

BEGIN
  IF assigned(FParent) THEN
  BEGIN
    Count := FParent.Count;
    AParent:=FParent;
    IF AValue < 0 THEN AValue := 0;
    IF AValue >= Count THEN AValue := Count-1;
    IF AValue <> Index THEN
    BEGIN
      IF assigned(FGLScene) THEN FGLScene.BeginUpdate;
      FParent.Remove(Self,False);
      AParent.Insert(AValue, Self);
      IF assigned(FGLScene) THEN FGLScene.EndUpdate;
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetParentComponent(Value: TComponent);

// it's a bit tricky, because the real parent of root objects was hidden while streaming

BEGIN
  IF assigned(FParent) THEN
  BEGIN
    FParent.Remove(Self,False);
    FParent:=NIL;
  END;
  // meant is a root object (directly connected to one of the root nodes)
  IF Value IS TGLScene THEN
  BEGIN
    IF Self IS TLightSource THEN
    BEGIN
      TGLScene(Value).LightSources.AddChild(Self);
      TLightSource(Self).LightID; // make the handle valid before any properties are assigned
    END
    ELSE
      IF Self IS TCamera THEN TGLScene(Value).Cameras.AddChild(Self) ELSE
        TGLScene(Value).Objects.AddChild(Self);
  END
  ELSE TComposite(Value).AddChild(Self);  // normal parent-child relation
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.MoveTo(NewParent: TComposite);

BEGIN
  IF assigned(FParent) THEN
  BEGIN
    FParent.Changed:=True;
    FParent.Remove(Self,False); // take all children with it (if object is a composite)
    FParent:=NIL;
  END;
  IF assigned(NewParent) THEN NewParent.AddChild(Self)
                         ELSE FGLScene:=NIL;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.PrepareList;

BEGIN
  glPushMatrix;
  glNewList(FHandle,GL_COMPILE);
  IF ocPosition IN FChanges THEN glTranslatef(FPosition.X,FPosition.Y,FPosition.Z);
  IF ocRotation IN FChanges THEN
  BEGIN
    glRotatef(FRotation[0],1,0,0);
    glRotatef(FRotation[1],0,1,0);
    glRotatef(FRotation[2],0,0,1);
  END;
  IF ocScaling   IN FChanges THEN glScalef(FScaling[0],FScaling[1],FScaling[2]);
  IF ocShininess IN FChanges THEN glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,FShininess);
  WITH FAmbient DO
    IF FChanged THEN glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,@FColor);
  WITH FDiffuse DO
    IF FChanged THEN glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,@FColor);
  WITH FSpecular DO
    IF FChanged THEN glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,@FColor);
  IF NOT FTexture.Bitmap.Empty THEN
  BEGIN
    glEnable(GL_TEXTURE_2D);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
    FTexture.Apply;
  END
  ELSE glDisable(GL_TEXTURE_2D);
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.FinishList;

BEGIN
  glEndList;
  glPopMatrix;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetAmbient(AValue: TGLColor);

BEGIN
  WITH FAmbient DO
  BEGIN
    Red:=AValue.Red;
    Green:=AValue.Green;
    Blue:=AValue.Blue;
    Alpha:=AValue.Alpha;
  END;
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetChanged(AValue: Boolean);

BEGIN
  IF FChanged <> AValue THEN
  BEGIN
    FChanged:=AValue;
    IF AValue AND assigned(FParent) THEN FParent.Changed:=True; 
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetCSGGroup(AGroup: TCSGGroup);

BEGIN
  IF AGroup <> FCSGGroup THEN
  BEGIN
    FCSGGroup:=AGroup;
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetDiffuse(AValue: TGLColor);

BEGIN
  WITH FDiffuse DO
  BEGIN
    Red:=AValue.Red;
    Green:=AValue.Green;
    Blue:=AValue.Blue;
    Alpha:=AValue.Alpha;
  END;
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetSpecular(AValue: TGLColor);

BEGIN
  WITH FSpecular DO
  BEGIN
    Red:=AValue.Red;
    Green:=AValue.Green;
    Blue:=AValue.Blue;
    Alpha:=AValue.Alpha;
  END;
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetShininess(AValue: GLFloat);

BEGIN
  IF FShininess <> AValue THEN
  BEGIN
    FShininess:=AValue;
    Include(FChanges,ocShininess);
    Changed:=True;
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetPosition(APosition: TGLCoordinates);

BEGIN
  FPosition:=APosition;
  Include(FChanges,ocPosition);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetPositionW(AValue: GLFloat);

BEGIN
  FPosition.W:=AValue;
  Include(FChanges,ocPosition);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetPositionX(AValue: GLFloat);

BEGIN
  FPosition.X:=AValue;
  Include(FChanges,ocPosition);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetPositionY(AValue: GLFloat);

BEGIN
  FPosition.Y:=AValue;
  Include(FChanges,ocPosition);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetPositionZ(AValue: GLFloat);

BEGIN
  FPosition.Z:=AValue;
  Include(FChanges,ocPosition);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetRotationX(AValue: GLFloat);

BEGIN
  IF AValue <> FRotation[0] THEN
  BEGIN
    FRotation[0]:=AValue;
    Include(FChanges,ocRotation);
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetRotationY(AValue: GLFloat);

BEGIN
  IF AValue <> FRotation[1] THEN
  BEGIN
    FRotation[1]:=AValue;
    Include(FChanges,ocRotation);
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.SetRotationZ(AValue: GLFloat);

BEGIN
  IF AValue <> FRotation[2] THEN
  BEGIN
    FRotation[2]:=AValue;
    Include(FChanges,ocRotation);
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.Translate(Tx,Ty,Tz : GLFloat);

BEGIN
  WITH FPosition DO
  BEGIN
    X:=Tx; Y:=Ty; Z:=Tz;
    Include(FChanges,ocPosition);
    Changed:=True;
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.Rotate(Rx,Ry,Rz : GLFloat);

BEGIN
  FRotation[0]:=Rx; FRotation[1]:=Ry; FRotation[2]:=Rz;
  Include(FChanges,ocRotation);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneObject.Scale(Sx,Sy,Sz : GLFloat);

BEGIN
  FScaling[0]:=Sx; FScaling[1]:=Sy; FScaling[2]:=Sz;
  Include(FChanges,ocScaling);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//---------------------- TCamera -----------------------------------------------

PROCEDURE TCamera.SetActive(AValue: Boolean);

VAR I : Integer;

BEGIN
  IF AValue <> FActive THEN
  BEGIN
    FActive:=AValue;
    IF FActive THEN
      WITH FGLScene,Cameras DO
        FOR I:=0 TO Count-1 DO
          IF Cameras[I] <> Self THEN TCamera(Cameras[I]).FActive:=False;
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//-------------------- TComposite ----------------------------------------------

CONSTRUCTOR TComposite.Create(AOwner: TComponent);

BEGIN
  INHERITED Create(AOwner);
  FChildren:=TList.Create;
  FCSGOperation:=A_OR_B;
END;

//------------------------------------------------------------------------------

DESTRUCTOR TComposite.Destroy;

BEGIN
  IF FChildren.Count > 0 THEN DeleteChildren;
  FChildren.Free;
  DestroyList;
  INHERITED Destroy;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.DeleteChildren;

BEGIN
  // children remove themself from child list
  WHILE FChildren.Count > 0 DO TSceneObject(FChildren.Items[0]).Free;
  FChildren.Clear;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.GetChildren(AProc: TGetChildProc);

VAR I : Integer;

BEGIN
  FOR I:=0 TO Count-1 DO AProc(FChildren[I]);
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.SetChildOrder(AChild: TComponent; Order: Integer);

BEGIN
  WITH FChildren DO
    IF TSceneObject(AChild).Index > -1 THEN Move(TSceneObject(AChild).Index,Order)
                                       ELSE Insert(Order,AChild);
END;

//------------------------------------------------------------------------------

FUNCTION TComposite.Get(Index: Integer): TSceneObject;

BEGIN
  Result:=FChildren[Index];
END;

//------------------------------------------------------------------------------

FUNCTION TComposite.GetCount: Integer;

BEGIN
  Result:=FChildren.Count;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.SetCSGOperation(AValue: TCSGOperation);

BEGIN
  IF AValue <> FCSGOperation THEN
  BEGIN
    FCSGOperation:=AValue;
    Changed:=True;
    IF assigned(FGLScene) THEN FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.AddChild(AChild: TSceneObject);

BEGIN
  FChildren.Add(AChild);
  AChild.FParent:=Self;
  AChild.FGLScene:=FGLScene;
  Changed:=True;
  IF assigned(FGLScene) THEN
  BEGIN
    FGLScene.Note(AChild,soAdd);
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

FUNCTION TComposite.AddNewChild(AChild: TSceneObjectClass): TSceneObject;

// Creates a new scene object and adds it to this composite object

BEGIN
  Result:=CreateSceneObject(FGLScene,AChild);
  AddChild(Result);
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.Insert(AIndex: Integer; AChild: TSceneObject);

BEGIN
  WITH FChildren DO
  BEGIN
    IF assigned(AChild.FParent) THEN AChild.FParent.Remove(AChild,False);
    Insert(AIndex,AChild);
  END;
  Changed:=True;
  AChild.FParent:=Self;
  IF AChild.FGLScene <> FGLScene THEN AChild.DestroyList;
  AChild.FGLScene:=FGLScene;
  IF assigned(FGLScene) THEN
  BEGIN
    FGLScene.Note(AChild,soAdd);
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.PrepareList;

// prepares all child display lists by simply calling their handles

VAR I : Integer;

BEGIN
  glPushMatrix;
  FOR I:=0 TO FChildren.Count-1 DO
    IF Children[I].Handle = 0 THEN {ShowError('Child handle couldn''t be created!')};
  glPopMatrix;
  // do the standard transformation
  INHERITED PrepareList;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.Remove(AChild: TSceneObject; KeepChildren: Boolean);

{Takes a scene object out of the child list, but doesn't destroy it. If 'KeepChildren'
 is true and 'AChild' is a composite, its children will be kept as new children in this
 composite.}

BEGIN
  FChildren.Remove(AChild);
  AChild.FParent:=NIL;
  IF KeepChildren AND (AChild IS TComposite) THEN
  BEGIN
    FGLScene.BeginUpdate;
    WITH TComposite(AChild) DO
      WHILE Count > 0 DO Children[0].MoveTo(Self);
    FGLScene.EndUpdate;
  END;
  Changed:=True;
  IF assigned(FGLScene) THEN
  BEGIN
    FGLScene.Note(AChild,soRemove);
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TComposite.BuildList;

VAR I,
    ACount,
    BCount  : Integer;
    AList,
    BList   : ARRAY[Byte] OF GLEnum;

BEGIN
  IF FChildren.Count = 0 THEN Exit;
  IF FChildren.Count = 1 THEN
  BEGIN
    glPushMatrix;
    glCallList(TSceneObject(FChildren[0]).Handle);
    glPopMatrix;
  END
  ELSE
  BEGIN
    ACount:=0;
    BCount:=0;
    FOR I:=0 TO FChildren.Count-1 DO
    CASE TSceneObject(FChildren[I]).CSGGroup OF
      CSG_A : BEGIN
                AList[ACount+1]:=TSceneObject(FChildren[I]).Handle;
                Inc(ACount);
              END;
      CSG_B : BEGIN
                BList[BCount+1]:=TSceneObject(FChildren[I]).Handle;
                Inc(BCount);
              END;
    END;
    AList[0]:=ACount;
    BList[0]:=BCount; 
    CASE FCSGOperation OF
      A_OR_B  : CSG_DoOR(AList,BList);
      A_AND_B : CSG_DoAND(AList,BList);
      A_SUB_B : CSG_DoSUB(AList,BList);
    END;
  END;
END;

//------------------ TLightSource ----------------------------------------------

CONSTRUCTOR TLightSource.Create(AOwner: TComponent);

BEGIN
  INHERITED Create(AOwner);
  FShining:=False;
  FConstAttenuation:=1;
  FLinearAttenuation:=0;
  FQuadraticAttenuation:=0;
  FSpotCutOff:=180;
  FSpotExponent:=0;
  FSpotDirection:=MakeAffineVector(0,0,-1);
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.DestroyList;

BEGIN
  Release;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.Release;

BEGIN
  IF assigned(FGLScene) AND
     (FHandle > 0)      AND
     NOT (csDestroying IN FGLScene.ComponentState) THEN
  BEGIN
    WITH FGLScene DO
    BEGIN
      RequestedState([stContextActive]);
      glDisable(FHandle);
    END;
    Changed:=True;
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

FUNCTION TLightSource.GetLightID: TObjectHandle;

// light sources have other handle types than normal scene objects

VAR I : Integer;

BEGIN
  IF (FHandle = 0) AND assigned(FParent) THEN
  BEGIN
    // the handle is created from the index (position) in the parent comp
    FHandle:=GL_LIGHT0+Index;            
  END;
  Result:=FHandle;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SwitchOn;

BEGIN
  IF NOT FShining THEN
  BEGIN
    FShining:=True;
    WITH FGLScene DO
    BEGIN
      RequestedState([stContextActive]);
      glEnable(LightID);
      CheckOpenGLError;
    END;
    Changed:=True;
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

procedure TLightSource.SwitchOff;

BEGIN
  IF FShining THEN
  BEGIN
    Fshining:=False;
    WITH FGLScene DO
    BEGIN
      RequestedState([stContextActive]);
      glDisable(LightID);
      CheckOpenGLError;
    END;
    Changed:=True;
    FGLScene.NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetShining(AValue: Boolean);

BEGIN
  IF AValue THEN SwitchOn
            ELSE SwitchOff;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetSpotDirection(AVector: TAffineFloatVector);

BEGIN
  FSpotDirection:=AVector;
  Include(FChanges,ocSpot);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetSpotExponent(AValue: GLFloat);

BEGIN
  FSpotExponent:=AValue;
  Include(FChanges,ocSpot);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetSpotCutOff(AValue: GLFloat);

BEGIN
  FSpotCutOff:=AValue;
  Include(FChanges,ocSpot);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetConstAttenuation(AValue: GLFloat);

BEGIN
  FConstAttenuation:=AValue;
  Include(FChanges,ocAttenuation);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetLinearAttenuation(AValue: GLFloat);

BEGIN
  FLinearAttenuation:=AValue;
  Include(FChanges,ocAttenuation);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TLightSource.SetQuadraticAttenuation(AValue: GLFloat);

BEGIN
  FQuadraticAttenuation:=AValue;
  Include(FChanges,ocAttenuation);
  Changed:=True;
  FGLScene.NotifyChange;
END;

//------------------ TGLScene --------------------------------------------------

PROCEDURE TGLScene.RegisterExtensions(Extensions: STRING);

{get and store proc addresses for the given OpenGL extensions}

BEGIN
  {IF Pos('GL_WIN_swap_hint',Extensions) > 0 THEN
    AddSwapRect:=wglGetProcAddress('GL_WIN_swap_hint');}
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.CreateParams(VAR Params: TCreateParams);

BEGIN
  INHERITED CreateParams(Params);
  WITH Params DO
  BEGIN
    IF NOT assigned(Parent)THEN WndParent:=GetDesktopWindow;
    ExStyle:=ExStyle+WS_EX_TRANSPARENT;
    WindowClass.Style:=WindowClass.Style+CS_OWNDC;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.AddNotifier(ASceneTree: TAbstractSceneTree);

BEGIN
  IF FNotifiers = NIL THEN FNotifiers:=TList.Create;
  IF FNotifiers.IndexOf(ASceneTree) < 0 THEN FNotifiers.Add(ASceneTree);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.GetChildren(AProc: TGetChildProc);

BEGIN
  FObjects.GetChildren(AProc);
  FLightSources.GetChildren(AProc);
  FCameras.GetChildren(AProc);
END;

//------------------------------------------------------------------------------

FUNCTION TGLScene.GetDeviceContext(VAR WindowHandle: HWnd): HDC;

BEGIN
  Result:=GetDC(Handle);
  IF Result = 0 THEN RAISE EOutOfResources.CreateRes(SWindowDCError);
  WindowHandle:=Handle;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.ReadContextProperties;

VAR IntProperty : GLInt;

BEGIN
  glGetIntegerv(GL_MAX_LIGHTS,@FMaxLightSources);
  IF glIsEnabled(GL_DEPTH_TEST) THEN Include(FCurrentStates,stDepthTest);
  IF glIsEnabled(GL_CULL_FACE)  THEN Include(FCurrentStates,stCullFace);
  IF glIsEnabled(GL_LIGHTING)   THEN Include(FCurrentStates,stLighting);
  glGetIntegerv(GL_FRONT_FACE,@IntProperty);

  //read OpenGL extensions
  RegisterExtensions(StrPas(PChar(glGetString(GL_EXTENSIONS))));
  glEnable(GL_NORMALIZE);
  glEnable(GL_CULL_FACE);
  glDepthFunc(GL_LESS);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,0);
END;

//------------------------------------------------------------------------------

FUNCTION TGLScene.GetRenderingContext: HGLRC;

// retrieves the current rendering context or creates a new one if not associated yet

BEGIN
  // context associated to window?
  IF FRenderingContext = 0 THEN
  BEGIN
    {// no -> context associated to current thread?
    FRenderingContext:=wglGetCurrentContext; // yes -> take this}
    IF FRenderingContext = 0 THEN
    BEGIN
      // no -> create a new context
      SetDCPixelFormat(Canvas.Handle);
      FRenderingContext:=wglCreateContext(Canvas.Handle);
      // still no rendering context? -> must be something wrong
      IF FRenderingContext = 0 THEN ShowError(ErrorNoContext);
    END;
    wglMakeCurrent(Canvas.Handle,FRenderingContext);
    ReadContextProperties;
    wglMakeCurrent(0,0);
  END;
  Result:=FRenderingContext;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetChildOrder(AChild: TComponent; Order: Integer);

BEGIN
  (AChild AS TSceneObject).Index:=Order;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetDCPixelFormat(Handle: HDC);

{set the properties requiered to draw to canvas}

CONST PFD_GENERIC_ACCELERATED = $1000; // declaration not in this version of 'windows.pas'

VAR PFDescriptor : TPixelFormatDescriptor;
    PixelFormat  : Integer;

BEGIN
  FillChar(PFDescriptor,SizeOf(PFDescriptor),0);
  WITH PFDescriptor DO
  BEGIN
    nSize     :=sizeof(PFDescriptor);  // Size of this structure
    nVersion  :=1;                     // Version number
    dwFlags   :=PFD_DRAW_TO_WINDOW OR
                PFD_SUPPORT_OPENGL OR
                PFD_DOUBLEBUFFER;
    iPixelType:=PFD_TYPE_RGBA;         // RGBA pixel values
    cColorBits:=32;                    // 32-bit color
    cDepthBits:=16;                    // 16-bit depth buffer
    cStencilBits:=4;                   // 4-bit stencil buffer
    iLayerType:=PFD_MAIN_PLANE;        // Layer type
  END;

  PixelFormat:=ChoosePixelFormat(Handle, @PFDescriptor);
  SetPixelFormat(Handle,PixelFormat,@PFDescriptor);
  // check the properties just set
  DescribePixelFormat(Handle,PixelFormat,SizeOf(PFDescriptor),PFDescriptor);
  WITH PFDescriptor DO
  BEGIN
    FDoubleBuffered:=(dwFlags AND PFD_DOUBLEBUFFER) > 0;
    FAccelerated:=(dwFlags AND PFD_GENERIC_ACCELERATED) > 0;
    FBuffers:=[];
    IF cColorBits > 0 THEN Include(FBuffers,buColor);
    IF cDepthBits > 0 THEN Include(FBuffers,buDepth);
    IF cStencilBits > 0 THEN Include(FBuffers,buStencil);
    IF cAccumBits > 0 THEN Include(FBuffers,buAccum);
  END;
END;

//------------------------------------------------------------------------------

CONSTRUCTOR TGLScene.Create(AOwner: TComponent);

VAR CurrentObject : TSceneObject;

BEGIN
  INHERITED Create(AOwner);
  ControlStyle:=ControlStyle-[csOpaque];
  Width:=300; Height:=300;
  FRenderingContext:=0;

  // root creation
  FObjects:=TComposite.Create(Self); FObjects.Name:='ObjectRoot';
  FObjects.FGLScene:=Self;
  FLightSources:=TComposite.Create(Self); FLightSources.Name:='LightSourceRoot';
  FLightSources.FGLScene:=Self;
  FCameras:=TComposite.Create(Self); FCameras.Name:='CameraRoot';
  FCameras.FGLScene:=Self;

  // initialize private state variables
  FBackground:=clBtnFace;
  FSceneChanges:=[scBackground];
  FLightChanged:=False;
  FFOVAngle:=30;
  FNearClipPlane:=1;
  FFarClipPlane:=100;
  WITH FViewPort DO
  BEGIN
    Left:=0; Top:=0; Width:=300; Height:=300;
  END;
  // performance check off
  FMonitoring:=False;
  FFramesPerSecond:=0;
  FFrames:=0;
  FTicks:=0;
END;

//------------------------------------------------------------------------------

DESTRUCTOR TGLScene.Destroy;

BEGIN
  // Clean up and terminate.
  UnnecessaryState([stContextValid]);
  INHERITED Destroy;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.Loaded;

BEGIN
  INHERITED Loaded;
  SetPerspective(FFOVAngle,FNearClipPlane,FFarClipPlane);
  SetViewPort(0,0,Width,Height);
  RequestedState([stContextActive]);
  glFrontFace(FaceWindingToNative[FFrontFaceWinding]);
  IF DepthTest THEN RequestedState([stDepthtest]);
  IF FaceCulling THEN RequestedState([stCullFace]);
  IF Lighting THEN RequestedState([stLighting]);
  UnnecessaryState([stContextActive]);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.WMEraseBkgnd(VAR Message: TWMEraseBkgnd);

BEGIN
  Message.Result:=1;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.WMSize(VAR Message: TWMSize);

BEGIN
  INHERITED;
  IF FRenderingContext > 0 THEN
  BEGIN
    // define viewport and perspective
    RequestedState([stContextActive]);
    WITH FViewPort DO
    BEGIN
      Width:=Message.Width;
      Height:=Message.Height;
      IF Height = 0 THEN Height:=1;
      glViewport(Left,Top,Width,Height);
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity;
      gluPerspective(FFOVAngle,Width/Height,FNearClipPlane,FFarClipPlane);
      UnnecessaryState([stContextActive]);
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.Paint;

BEGIN
  DrawScene;
  IF csDesigning IN ComponentState THEN Canvas.DrawFocusRect(ClientRect);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetPerspective(AFOVAngle,ANearPlane,AFarPlane: GLFloat);

BEGIN
  FFOVAngle:=AFOVAngle;
  FNearClipPlane:=ANearPlane;
  FFarClipPlane:=AFarPlane;
  Perform(WM_SIZE,SIZE_RESTORED,MakeLong(Height,Width));
  NotifyChange;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetViewPort(X,Y,W,H: Integer);

BEGIN
  WITH FViewPort DO
  BEGIN
    Left:=X; Top:=Y; Width:=W; Height:=H;
  END;
  Perform(WM_SIZE,SIZE_RESTORED,MakeLong(Height,Width));
  NotifyChange;
END;

//------------------------------------------------------------------------------

FUNCTION TGLScene.IsUpdating: Boolean;

BEGIN
  Result:=(FUpdateCount <> 0) OR (csLoading IN ComponentState) OR (csDestroying IN ComponentState);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.BeginUpdate;

BEGIN
  Inc(FUpdateCount);
  RequestedState([stContextActive]);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.EndUpdate;

BEGIN
  IF FUpdateCount > 0 THEN Dec(FUpdateCount);
  IF FUpdateCount = 0 THEN Refresh;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.Note(AObject: TSceneObject; Operation: TSceneOperation);

// notifies designers of the scene about design relevant changes

VAR I : Integer;

BEGIN
  IF assigned(FNotifiers) AND NOT (csDestroying IN ComponentState) THEN
    FOR I:=0 TO FNotifiers.Count-1 DO TAbstractSceneTree(FNotifiers[I]).Notify(AObject,Operation);
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.RemoveNotifier(ASceneTree: TAbstractSceneTree);

BEGIN
  IF assigned(FNotifiers) THEN
  BEGIN
    FNotifiers.Remove(ASceneTree);
    IF FNotifiers.Count = 0 THEN FNotifiers.Free;
    FNotifiers:=NIL;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.NotifyChange;

BEGIN
  IF NOT IsUpdating THEN Refresh;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.RequestedState(States: TGLStates);

VAR NeededStates : TGLStates;

BEGIN
  // get all states, which are requested but not yet set
  NeededStates:=States-FCurrentStates;
  IF NeededStates <> [] THEN
  BEGIN
    // now set all unset states
    IF (stContextValid IN NeededStates) OR (stContextActive IN NeededStates) THEN
    BEGIN
      IF RenderingContext > 0 THEN
      BEGIN
        IF stContextActive IN NeededStates THEN wglMakeCurrent(Canvas.Handle,FRenderingContext)
                                           ELSE
                                           BEGIN
                                             Exclude(FCurrentStates,stContextActive);
                                             Exclude(NeededStates,stContextActive);
                                           END;
        Include(FCurrentStates,stContextValid);
      END
      ELSE
      BEGIN
        Exclude(FCurrentStates,stContextValid);
        Exclude(NeededStates,stContextValid);
      END;
    END;  
    IF stAlphaTest      IN NeededStates THEN glEnable(GL_ALPHA_TEST);
    IF stAutoNormal     IN NeededStates THEN glEnable(GL_AUTO_NORMAL);
    IF stBlend          IN NeededStates THEN glEnable(GL_BLEND);
    IF stColorMaterial  IN NeededStates THEN glEnable(GL_COLOR_MATERIAL);
    IF stCullFace       IN NeededStates THEN glEnable(GL_CULL_FACE);
    IF stDepthTest      IN NeededStates THEN glEnable(GL_DEPTH_TEST);
    IF stDither         IN NeededStates THEN glEnable(GL_DITHER);
    IF stFog            IN NeededStates THEN glEnable(GL_FOG);
    IF stLighting       IN NeededStates THEN glEnable(GL_LIGHTING);
    IF stLineSmooth     IN NeededStates THEN glEnable(GL_LINE_SMOOTH);
    IF stLineStipple    IN NeededStates THEN glEnable(GL_LINE_STIPPLE);
    IF stLogicOp        IN NeededStates THEN glEnable(GL_LOGIC_OP);
    IF stNormalize      IN NeededStates THEN glEnable(GL_NORMALIZE);
    IF stPointSmooth    IN NeededStates THEN glEnable(GL_POINT_SMOOTH);
    IF stPolygonSmooth  IN NeededStates THEN glEnable(GL_POLYGON_SMOOTH);
    IF stPolygonStipple IN NeededStates THEN glEnable(GL_POLYGON_STIPPLE);
    IF stScissorTest    IN NeededStates THEN glEnable(GL_SCISSOR_TEST);
    IF stStencilTest    IN NeededStates THEN glEnable(GL_STENCIL_TEST);
    FCurrentStates:=FCurrentStates+NeededStates;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.UnnecessaryState(States: TGLStates);

VAR TakeOutStates : TGLStates;

BEGIN
  // get all states, which are to be taken out, but still set
  TakeOutStates:=States*FCurrentStates;
  IF TakeOutStates <> [] THEN
  BEGIN
    // now reset all these states
    IF stAlphaTest      IN TakeOutStates THEN glDisable(GL_ALPHA_TEST);
    IF stAutoNormal     IN TakeOutStates THEN glDisable(GL_AUTO_NORMAL);
    IF stBlend          IN TakeOutStates THEN glDisable(GL_BLEND);
    IF stColorMaterial  IN TakeOutStates THEN glDisable(GL_COLOR_MATERIAL);
    IF stCullFace       IN TakeOutStates THEN glDisable(GL_CULL_FACE);
    IF stDepthTest      IN TakeOutStates THEN glDisable(GL_DEPTH_TEST);
    IF stDither         IN TakeOutStates THEN glDisable(GL_DITHER);
    IF stFog            IN TakeOutStates THEN glDisable(GL_FOG);
    IF stLighting       IN TakeOutStates THEN glDisable(GL_LIGHTING);
    IF stLineSmooth     IN TakeOutStates THEN glDisable(GL_LINE_SMOOTH);
    IF stLineStipple    IN TakeOutStates THEN glDisable(GL_LINE_STIPPLE);
    IF stLogicOp        IN TakeOutStates THEN glDisable(GL_LOGIC_OP);
    IF stNormalize      IN TakeOutStates THEN glDisable(GL_NORMALIZE);
    IF stPointSmooth    IN TakeOutStates THEN glDisable(GL_POINT_SMOOTH);
    IF stPolygonSmooth  IN TakeOutStates THEN glDisable(GL_POLYGON_SMOOTH);
    IF stPolygonStipple IN TakeOutStates THEN glDisable(GL_POLYGON_STIPPLE);
    IF stScissorTest    IN TakeOutStates THEN glDisable(GL_SCISSOR_TEST);
    IF stStencilTest    IN TakeOutStates THEN glDisable(GL_STENCIL_TEST);

    IF (stContextValid  IN TakeOutStates) OR (stContextActive IN TakeOutStates) THEN
    BEGIN
      wglMakeCurrent(0,0);
      Include(TakeOutStates,stContextActive);
      IF stContextValid IN TakeOutStates THEN
      BEGIN
        IF FRenderingContext > 0 THEN wglDeleteContext(FRenderingContext);
        FRenderingContext:=0;
        TakeOutStates:=GLAllStates;
      END;
    END;
    FCurrentStates:=FCurrentStates-TakeOutStates;
  END;
END;

//------------------------------------------------------------------------------

FUNCTION GetClearMask(Buffers: TBuffers): GLBitField;

BEGIN
  Result:=0;
  IF buColor   IN Buffers THEN Result:=Result OR GL_COLOR_BUFFER_BIT;
  IF buDepth   IN Buffers THEN Result:=Result OR GL_DEPTH_BUFFER_BIT;
  IF buStencil IN Buffers THEN Result:=Result OR GL_STENCIL_BUFFER_BIT;
  IF buAccum   IN Buffers THEN Result:=Result OR GL_ACCUM_BUFFER_BIT;
//  IF buAux IN Buffers THEN Result:=Result OR GL_AUX_BUFFER_BIT;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.DrawScene;

VAR Ticks : Longint;
    r,g,b : GLFloat;

BEGIN
  // performance data demanded?
  IF FMonitoring THEN Ticks:=GetTickCount; // yes, take start clicks
  RequestedState([stContextActive]);
  AdjustLightSources;

  IF FSceneChanges <> [] THEN
  BEGIN
    IF scBackground IN FSceneChanges THEN
    BEGIN
      ConvertWinColor(FBackground,r,g,b);
      glClearColor(r,g,b,1);
    END;
    FSceneChanges:=[];
  END;

  // cear the buffers
  glClear(GetClearMask(Buffers));

  // define the modelview transformation
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity;
  ApplyCamera;
  glCallList(Objects.Handle);
  glFinish;
  IF FDoubleBuffered THEN wglSwapBuffers(Canvas.Handle);
  CheckOpenGLError;

  // performance data demanded?
  IF FMonitoring THEN
  BEGIN
    // yes, calculate frames per second
    Inc(FFrames);
    Inc(FTicks,GetTickCount-Ticks);
    // hand tuned (1 tick = 1.8 s), seems more realistic than 1 tick = 1 ms
    IF FTicks > 0 THEN FFramesPerSecond:=555.5*FFrames/FTicks;
  END;
  UnnecessaryState([stContextActive]);
  Objects.Changed:=False;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.AdjustLightSources;

// updates all used and modified light sources

VAR I  : Integer;
    LS : TLightSource;

BEGIN
  // start searching through all light sources
  FOR I:=0 TO LightSources.Count-1 DO
  BEGIN
    LS:=TLightSource(LightSources[I]);
    IF LS.Shining THEN
    WITH LS DO
    BEGIN
      IF ocPosition IN FChanges THEN
      BEGIN
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity;
        glLightfv(LightID,GL_POSITION,@FPosition);
        Exclude(FChanges,ocPosition);
      END;
      WITH FAmbient  DO IF FChanged THEN glLightfv(LightID,GL_AMBIENT,@FColor);
      WITH FDiffuse  DO IF FChanged THEN glLightfv(LightID,GL_DIFFUSE,@FColor);
      WITH FSpecular DO IF FChanged THEN glLightfv(LightID,GL_SPECULAR,@FColor);
      IF ocSpot IN FChanges THEN
      BEGIN
        glLightfv(LightID,GL_SPOT_DIRECTION,@FSpotDirection);
        glLightfv(LightID,GL_SPOT_EXPONENT,@FSpotExponent);
        glLightfv(LightID,GL_SPOT_CUTOFF,@FSpotCutOff);
        Exclude(FChanges,ocSpot);
      END;
      IF ocAttenuation IN FChanges THEN
      BEGIN
        glLightfv(LightID,GL_CONSTANT_ATTENUATION,@FConstAttenuation);
        glLightfv(LightID,GL_LINEAR_ATTENUATION,@FLinearAttenuation);
        glLightfv(LightID,GL_QUADRATIC_ATTENUATION,@FQuadraticAttenuation);
        Exclude(FChanges,ocAttenuation);
      END;
    END;
  END;
  CheckOpenGLError;
  FLightSources.Changed:=False;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.ApplyCamera;

VAR I             : Integer;
    CurrentCamera : TCamera;

BEGIN
  CurrentCamera:=NIL;
  FOR I:=0 TO Cameras.Count-1 DO
    IF TCamera(Cameras[I]).Active THEN
    BEGIN
      CurrentCamera:=Cameras[I] AS TCamera;
      Break;
    END;
  IF assigned(CurrentCamera) THEN
  WITH CurrentCamera DO
  BEGIN
    gluLookAt(X,Y,Z,0,0,-1,0,1,0);
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetBackground(AColor: TColor);

BEGIN
  IF FBackground <> AColor THEN
  BEGIN
    FBackground:=AColor;
    Include(FSceneChanges,scBackground);
    NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetFrontFaceWinding(AValue: TFaceWinding);

BEGIN
  IF FFrontFaceWinding <> AValue THEN
  BEGIN
    FFrontFaceWinding:=AValue;
    IF NOT IsUpdating THEN
    BEGIN
      RequestedState([stContextActive]);
      glFrontFace(FaceWindingToNative[AValue]);
    END;
    NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetDepthTest(AValue: Boolean);

BEGIN
  IF FDepthTest <> AValue THEN
  BEGIN
    FDepthTest:=AValue;
    IF NOT IsUpdating THEN
    BEGIN
      RequestedState([stContextActive]);
      IF AValue THEN RequestedState([stDepthTest])
                ELSE UnnecessaryState([stDepthTest]);
    END;
    NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetFaceCulling(AValue: Boolean);

BEGIN
  IF FFaceCulling <> AValue THEN
  BEGIN
    FFaceCulling:=AValue;
    IF NOT IsUpdating THEN
    BEGIN
      RequestedState([stContextActive]);
      IF AValue THEN RequestedState([stCullFace])
                ELSE UnnecessaryState([stCullFace]);
    END;
    NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TGLScene.SetLighting(AValue: Boolean);

BEGIN
  IF FLighting <> AValue THEN
  BEGIN
    FLighting:=AValue;
    IF NOT IsUpdating THEN
    BEGIN
      RequestedState([stContextActive]);
      IF AValue THEN RequestedState([stLighting])
                ELSE UnnecessaryState([stLighting]);
    END;
    NotifyChange;
  END;
END;

//------------------------------------------------------------------------------

BEGIN
  // determine user default language for localized messages
  CASE GetUserDefaultLangID AND $3FF OF
    LANG_GERMAN  : LangOffset:=1000;
    LANG_ITALIAN : LangOffset:=2000
  ELSE LangOffset:=0;
  END;
  LangOffset:=0;
END.
